/*
 * adc_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * The vADCTask task first configures the ADC 0 peripheral of the TM4C123GH6PM
 * MCU to sample on channel 0 (PE3) and channel 1 (PE2) on two different
 * sequencers.  Two tasks are created so each task is responsible for each
 * sequencer.  The task processes the conversion data for its sequencer and
 * then passes the final data to a third task through a queue.  The third task
 * receives any data from the queue and displays the data on a terminal window.
 *
 * The ISR for ADC 0 Sequence 0 is xADCSeq0Handler and it is linked to the
 * interrupt vector table for the TM4C123GH6PM MCU in the startup_ccs.c file.
 * Similarly, the ISR for ADC 0 Sequence 1 is xADCSeq1Handler.
 * Instead of manually linking the ISR, the IntRegister API can be used
 * instead.  The ISR clears the hardware interrupt flag and then uses the
 * FreeRTOS Task Notify feature to defer further processing to a task.  This
 * makes for a very lean ISR which is recommended, especially for hardware
 * interrupts.
 *
 * The prvSeq0ReceiveTask waits for the notification from the ISR.  The
 * notification is given by the ISR when the DMA completes the transfer from
 * ADC to the buffer. It processes the data in the buffer and then send the
 * processed data to a queue.  prvSeq1ReceiveTask does the same thing
 * independent for sequencer 1 and passes the data to the queue.  In real
 * life applications the two tasks can take different amount of time to
 * process the data and the notifications they get from the ISRs can be
 * asynchronous to each other.
 *
 * The prvQueueReceiveTask waits for any data in the queue to unblock itself.
 * Once unblocked it merely displays the received data on the terminal window.
 * The queue is implemented to pass data by value.  Passing data by pointer
 * can also be employed.  The data type to pass on the queue is a simple
 * structure that stores two elements: an ID element that indicates the channel
 * number and the data element that stores the conversion data.
 *
 * This example uses AIN0 which is mapped to PE3 and AIN1 which is mapped to
 * PE2 as the ADC input channels.  Supply the test voltages to these inputs.
 *
 * This example uses UARTprintf for output of UART messages.  UARTprintf is not
 * a thread-safe API and is only being used for simplicity of the demonstration
 * and in a controlled manner.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_adc.h"
#include "inc/hw_types.h"
#include "inc/hw_udma.h"
#include "driverlib/adc.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/interrupt.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "driverlib/udma.h"
#include "utils/uartstdio.h"
/*-----------------------------------------------------------*/

/*
 * The number of items the queue can hold.  This is 4 as the receive task
 * will remove items as they are added, meaning the send task should always find
 * the queue empty.
 */
#define mainQUEUE_LENGTH                    ( 4 )

/*
 * Define the size of the ADC buffers.
 */
#define ADC_SAMPLE_BUF_SIZE 128

/*
 * Global buffers to store ADC sample data.
 */
static uint16_t pui16ADCBuffer1[ADC_SAMPLE_BUF_SIZE];
static uint16_t pui16ADCBuffer2[ADC_SAMPLE_BUF_SIZE];

/*
 * Declare a variable that is used to hold the handle of the ADC
 * interrupt task.
 */
TaskHandle_t xADCSeq0Handle = NULL;
TaskHandle_t xADCSeq1Handle = NULL;

/*
 * Queue used to send and receive complete struct AMessage structures.
 */
QueueHandle_t xStructQueue = NULL;

/* The queue used by both tasks. */
struct AMessage
{
    uint32_t ulAdcID;
    uint32_t ulAdcData;
} xMessage;

/*
 * The control table used by the uDMA controller.  This table must be aligned
 * to a 1024 byte boundary.
 */
#if defined(ewarm)
#pragma data_alignment=1024
uint8_t pui8ControlTable[1024];
#elif defined(ccs)
#pragma DATA_ALIGN(pui8ControlTable, 1024)
uint8_t pui8ControlTable[1024];
#else
uint8_t pui8ControlTable[1024] __attribute__ ((aligned(1024)));
#endif

/*
 * The count of uDMA errors.  This value is incremented by the uDMA error
 * handler.
 */
static uint32_t g_ui32uDMAErrCount = 0;

/*
 * The count of times the uDMA interrupt occurred but the uDMA transfer was not
 * complete.  This should remain 0.
 */
static uint32_t g_ui32BadISR = 0;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvSeq0ReceiveTask( void *pvParameters );
static void prvSeq1ReceiveTask( void *pvParameters );
static void prvQueueReceiveTask1( void *pvParameters );

/*
 * Called by main() to create the ADC task.
 */
void vADCTask( void );

/*
 * Hardware configuration for the ADC to configure channel, sequencer,
 * and interrupts.
 */
static void prvConfigureADC( void );

/*
 * Hardware configuration for the uDMA to configure channels and transfer
 * types.
 */
static void prvConfigureuDMA( void );
/*-----------------------------------------------------------*/

void vADCTask( void )
{
    /* Configure the ADC to use Channel 0 on Sequencer 0, be timer triggered,
     * and to fire an interrupt at the end of each sample sequence. */
    prvConfigureADC();

    /* Configure the uDMA for ADC channel 0 / 1 in basic mode and point the
     * uDMA to the addresses of the defined ADC buffers for each channel. */
    prvConfigureuDMA();

    /* Create a queue that can hold a maximum of 2 pointers with each being
     * 32-bit words. */
    xStructQueue = xQueueCreate( mainQUEUE_LENGTH, sizeof( xMessage ) );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for ADC Sequence 0 data processing task - for debug
     *    only as it is not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvSeq0ReceiveTask,
                 "Seq0",
                 configMINIMAL_STACK_SIZE + ADC_SAMPLE_BUF_SIZE*2,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 &xADCSeq0Handle );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for ADC Sequence 1 data processing task - for debug
     *    only as it is not used by the kernel.
     *  - The size of the stack to allocate to the task.  Since this task is
     *    creating multiple buffers that are used to store data, the stack size
     *    must be increased to account for the size of these buffers or else
     *    the task would run out of memory.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.  Because it is interrupt driven,
     *    this task should preempt the data processing task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvSeq1ReceiveTask,
                 "Seq1",
                 configMINIMAL_STACK_SIZE + ADC_SAMPLE_BUF_SIZE*2,
                 NULL,
                 tskIDLE_PRIORITY + 2,
                 &xADCSeq1Handle );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for queue receive task - for debug only as it is not
     *    used by the kernel.
     *  - The size of the stack to allocate to the task.  Since this task is
     *    creating multiple buffers that are used to store data, the stack size
     *    must be increased to account for the size of these buffers or else
     *    the task would run out of memory.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.  Because it is interrupt driven,
     *    this task should preempt the data processing task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvQueueReceiveTask1,
                 "Queue task",
                 configMINIMAL_STACK_SIZE + ADC_SAMPLE_BUF_SIZE*2,
                 NULL,
                 tskIDLE_PRIORITY + 2,
                 NULL );

    /* Initialize the UART and write initial status. */
    UARTprintf("Timer->ADC->uDMA demo!\n\n");

    /* Enables DMA channel so it can perform transfers.  As soon as the
     * channels are enabled, the peripheral will issue a transfer request
     * and the data transfers will begin. */
    uDMAChannelEnable(UDMA_CHANNEL_ADC0);
    uDMAChannelEnable(UDMA_CHANNEL_ADC1);
}
/*-----------------------------------------------------------*/

static void prvSeq0ReceiveTask( void *pvParameters )
{
unsigned long ulEventsToProcess;
uint32_t ui32Count, ui32AverageResult1;
uint32_t ui32SamplesTaken = 0;
struct AMessage xMessage;


    for( ;; )
    {
        /* Wait to receive a notification sent directly to this task from the
         * interrupt service routine. */
        ulEventsToProcess = ulTaskNotifyTake( pdTRUE, portMAX_DELAY );

        if (ulEventsToProcess != 0)
        {
            /* Process the data in pui16ADCBuffer1 and clear buffer entries. */
            ui32AverageResult1 = 0;

            for(ui32Count = 0; ui32Count < ADC_SAMPLE_BUF_SIZE; ui32Count++)
            {
                ui32AverageResult1 += pui16ADCBuffer1[ui32Count];
                pui16ADCBuffer1[ui32Count] = 0;
            }

            /* Enable for another uDMA block transfer. */
            uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT,
                                   UDMA_MODE_BASIC,
                                   (void *)(ADC0_BASE + ADC_O_SSFIFO0),
                                   &pui16ADCBuffer1, ADC_SAMPLE_BUF_SIZE);

            /* Enable DMA channel so it can perform transfers. */
            uDMAChannelEnable(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT);

            /* Track the number of samples taken and update the average. */
            ui32SamplesTaken += ADC_SAMPLE_BUF_SIZE;
            ui32AverageResult1 = ((ui32AverageResult1 ) /
                    ADC_SAMPLE_BUF_SIZE);

            xMessage.ulAdcID = 0;
            xMessage.ulAdcData = ui32AverageResult1;

            /* Send the entire structure by value to the queue */
            xQueueSend(
                    /* The handle of the queue. */
                    xStructQueue,
                    /* The address of the xMessage variable.
                     * sizeof( struct AMessage ) bytes are copied from here
                     * into the queue. */
                    ( void * ) &xMessage,
                    /* Block time of 0 says don't block if the queue is already
                     * full.  Check the value returned by xQueueSend() to know
                     * if the message was sent to the queue successfully. */
                    ( TickType_t ) 0 );
        }
        else
        {
            /* Timed out. */
        }
    }
}
/*-----------------------------------------------------------*/

static void prvSeq1ReceiveTask( void *pvParameters )
{
unsigned long ulEventsToProcess;
uint32_t ui32Count, ui32AverageResult2;
uint32_t ui32SamplesTaken = 0;
struct AMessage xMessage;

    for( ;; )
    {
        /* Wait to receive a notification sent directly to this task from the
         * interrupt service routine. */
        ulEventsToProcess = ulTaskNotifyTake( pdTRUE, portMAX_DELAY );

        if (ulEventsToProcess != 0)
        {
            /* Process the data in pui16ADCBuffer2 and clear buffer entries. */
            ui32AverageResult2 = 0;

            for(ui32Count = 0; ui32Count < ADC_SAMPLE_BUF_SIZE; ui32Count++)
            {
                ui32AverageResult2 += pui16ADCBuffer2[ui32Count];
                pui16ADCBuffer2[ui32Count] = 0;
            }

            /* Enable for another uDMA block transfer. */
            uDMAChannelTransferSet(UDMA_CHANNEL_ADC1 | UDMA_PRI_SELECT,
                                   UDMA_MODE_BASIC,
                                   (void *)(ADC0_BASE + ADC_O_SSFIFO1),
                                   &pui16ADCBuffer2, ADC_SAMPLE_BUF_SIZE);

            /* Enable DMA channel so it can perform transfers. */
            uDMAChannelEnable(UDMA_CHANNEL_ADC1 | UDMA_PRI_SELECT);

            /* Track the number of samples taken and update the average. */
            ui32SamplesTaken += ADC_SAMPLE_BUF_SIZE;
            ui32AverageResult2 = ((ui32AverageResult2 ) /
                    ADC_SAMPLE_BUF_SIZE);

            xMessage.ulAdcID = 1;
            xMessage.ulAdcData = ui32AverageResult2;

            /* Send the entire structure by value to the queue */
            xQueueSend( /* The handle of the queue. */
                    xStructQueue,
                    /* The address of the xMessage variable.
                     * sizeof( struct AMessage ) bytes are copied from here
                     * into the queue. */
                    ( void * ) &xMessage,
                    /* Block time of 0 says don't block if the queue is already
                     * full.  Check the value returned by xQueueSend() to know
                     * if the message was sent to the queue successfully. */
                    ( TickType_t ) 0 );
        }
        else
        {
            /* Timed out. */
        }
    }
}
/*-----------------------------------------------------------*/

static void prvQueueReceiveTask1( void *pvParameters )
{
struct AMessage xRxedStructure;

    for (;;)
    {
        if( xStructQueue != NULL )
        {
            /* Receive a message from the created queue to hold complex struct
             * AMessage structure.  Block for 10 ticks if a message is not
             * immediately available.  The value is read into a struct AMessage
             * variable, so after calling xQueueReceive(), xRxedStructure will
             * hold a copy of xMessage. */
            if( xQueueReceive( xStructQueue,
                               &( xRxedStructure ),
                               ( TickType_t ) 10 ) == pdPASS )
            {

                /* xRxedStructure now contains a copy of xMessage. */
                UARTprintf("Channel %d: %d\n", xRxedStructure.ulAdcID,
                                               xRxedStructure.ulAdcData);
            }
        }
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureADC( void )
{
    /* Enable the peripherals used by this application. */
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_ADC0);
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);

    /* Enable the GPIO pin for ADC0 Channel 0/1 (PE3/PE2) and configure it for
     * analog functionality. */
    GPIOPinTypeADC(GPIO_PORTE_BASE, GPIO_PIN_3);
    GPIOPinTypeADC(GPIO_PORTE_BASE, GPIO_PIN_2);


    /* Enable the peripherals used by this application. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_ADC0);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_ADC0))
    {
    }

    /* Disable interrupts for ADC0 sample sequence 0/1 to configure it. */
    ADCIntDisable(ADC0_BASE, 0);
    ADCIntDisable(ADC0_BASE, 1);


    /* Disable ADC0 sample sequence 0/1.  With the sequence disabled, it is now
     * safe to load the new configuration parameters. */
    ADCSequenceDisable(ADC0_BASE, 0);
    ADCSequenceDisable(ADC0_BASE, 1);


    /* Configure the source clock for ADC0 module */
    ADCClockConfigSet(ADC0_BASE, ADC_CLOCK_SRC_PIOSC | ADC_CLOCK_RATE_FULL, 1);

    /* Enable sample sequence 0/1 with a timer trigger.  Sequence 0/1 will
     * start the next conversion each time the timer times out. */
    ADCSequenceConfigure(ADC0_BASE, 0, ADC_TRIGGER_TIMER, 0);
    ADCSequenceConfigure(ADC0_BASE, 1, ADC_TRIGGER_TIMER, 0);


    /* Configure step 0 on sequence 0/1. There are two sequencers setup with
     * each sequencer taking sample for one channel.  Map channel 0
     * (ADC_CTL_CH0) to sequencer0 and channel 1 (ADC_CTL_CH1) to sequencer 1
     * in single-ended mode (default) and configure the interrupt flag
     * (ADC_CTL_IE) to be set when the sample is done.  Tell the ADC logic
     * that this is the last conversion on sequence 0/1 (ADC_CTL_END). */
    ADCSequenceStepConfigure(ADC0_BASE, 0, 0, ADC_CTL_CH0 | ADC_CTL_IE |
                             ADC_CTL_END);
    ADCSequenceStepConfigure(ADC0_BASE, 1, 0, ADC_CTL_CH1 | ADC_CTL_IE |
                             ADC_CTL_END);

    /* Since sample sequence 0/1 is now configured, it must be enabled. */
    ADCSequenceEnable(ADC0_BASE, 0);
    ADCSequenceEnable(ADC0_BASE, 1);


    /* Clear the interrupt status flag.  This is done to make sure the
     * interrupt flag is cleared before we sample. */
    ADCIntClear(ADC0_BASE, 0);
    ADCIntClear(ADC0_BASE, 1);


    /* Enables the DMA channel for the ADC0 sample sequence 0/1. */
    ADCSequenceDMAEnable(ADC0_BASE, 0);
    ADCSequenceDMAEnable(ADC0_BASE, 1);


    /* Enable the ADC 0 sample sequence 0/1 interrupt. */
    ADCIntEnable(ADC0_BASE, 0);
    ADCIntEnable(ADC0_BASE, 1);


    /* Enable the interrupt for ADC0 sequence 0/1 on the processor (NVIC). */
    IntEnable(INT_ADC0SS0);
    IntEnable(INT_ADC0SS1);

}
/*-----------------------------------------------------------*/

static void prvConfigureuDMA( void )
{
    /* Enable the peripherals used by this application. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UDMA);

    /* Enable the uDMA controller. */
    uDMAEnable();

    /* Point at the control table to use for channel control structures. */
    uDMAControlBaseSet(pui8ControlTable);

    /* Configure the control parameters for the primary control structure for
     * the ADC0 Sequence 0. The transfer data size is 16 bits, the
     * source address does not increment since it will be reading from a
     * register.  The destination address increment is 16-bits.  The
     * arbitration size is set to one byte transfers. */
    uDMAChannelControlSet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT, UDMA_SIZE_16 |
                          UDMA_SRC_INC_NONE | UDMA_DST_INC_16 | UDMA_ARB_1);

    /* Configure the control parameters for the primary control structure for
     * the ADC0 Sequence 1. The transfer data size is 16 bits, the
     * source address does not increment since it will be reading from a
     * register.  The destination address increment is 16-bits.  The
     * arbitration size is set to one byte transfers. */
    uDMAChannelControlSet(UDMA_CHANNEL_ADC1 | UDMA_PRI_SELECT, UDMA_SIZE_16 |
                          UDMA_SRC_INC_NONE | UDMA_DST_INC_16 | UDMA_ARB_1);

    /* Set up the transfer parameters for the ADC0 primary control structure
     * The mode is set to BASIC, the transfer source is the ADC Sample
     * Sequence Result FIFO 0 register, and the destination is the receive
     * buffer.  The transfer size is set to match the size of the buffer. */
    uDMAChannelTransferSet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT,
                           UDMA_MODE_BASIC,
                           (void *)(ADC0_BASE + ADC_O_SSFIFO0),
                           &pui16ADCBuffer1, ADC_SAMPLE_BUF_SIZE);

    /* Set up the transfer parameters for the ADC0 primary control structure
     * The mode is set to BASIC, the transfer source is the ADC Sample
     * Sequence Result FIFO 1 register, and the destination is the receive
     *  buffer.  The transfer size is set to match the size of the buffer. */
    uDMAChannelTransferSet(UDMA_CHANNEL_ADC1 | UDMA_PRI_SELECT,
                           UDMA_MODE_BASIC,
                           (void *)(ADC0_BASE + ADC_O_SSFIFO1),
                           &pui16ADCBuffer2, ADC_SAMPLE_BUF_SIZE);

    /* Set the USEBURST attribute for the uDMA ADC0/ADC1 channels.  This will
     * force the controller to always use a burst when transferring data from
     * the ADC to buffers.  This is somewhat more efficient bus usage than
     * the default which allows single or burst transfers. */
    uDMAChannelAttributeEnable(UDMA_CHANNEL_ADC0, UDMA_ATTR_USEBURST);
    uDMAChannelAttributeEnable(UDMA_CHANNEL_ADC1, UDMA_ATTR_USEBURST);
}
/*-----------------------------------------------------------*/

void xADCSeq0Handler( void )
{
BaseType_t xADCTaskWoken;

    /* Clear the hardware interrupt flag for ADC0. */
    ADCIntClear(ADC0_BASE, 0);

    /* The xADCTaskWoken parameter must be initialized to pdFALSE as
     * it will get set to pdTRUE inside the interrupt safe API function if a
     * context switch is required. */
    xADCTaskWoken = pdFALSE;

    if (uDMAChannelModeGet(UDMA_CHANNEL_ADC0 | UDMA_PRI_SELECT) ==
            UDMA_MODE_STOP)
    {
        /* Defer the interrupt processing to a Task to minimize time spent
         * within the hardware interrupt service routine.  Send a notification
         * directly to the task to which interrupt processing is being
         * deferred. */
        vTaskNotifyGiveFromISR( xADCSeq0Handle, &xADCTaskWoken );

        /* This FreeRTOS API call will handle the context switch if it is
         * required or have no effect if that is not needed. */
        portYIELD_FROM_ISR( xADCTaskWoken );
    }

}
/*-----------------------------------------------------------*/

void xADCSeq1Handler( void )
{
BaseType_t xADCTaskWoken;

    /* Clear the hardware interrupt flag for ADC0. */
    ADCIntClear(ADC0_BASE, 1);

    /* The xADCTaskWoken parameter must be initialized to pdFALSE as
     * it will get set to pdTRUE inside the interrupt safe API function if a
     * context switch is required. */
    xADCTaskWoken = pdFALSE;

    if (uDMAChannelModeGet(UDMA_CHANNEL_ADC1 | UDMA_PRI_SELECT) ==
            UDMA_MODE_STOP)
    {
        /* Defer the interrupt processing to a Task to minimize time spent
         * within the hardware interrupt service routine.  Send a notification
         * directly to the task to which interrupt processing is being
         * deferred. */
        vTaskNotifyGiveFromISR( xADCSeq1Handle, &xADCTaskWoken );

        /* This FreeRTOS API call will handle the context switch if it is
         * required or have no effect if that is not needed. */
        portYIELD_FROM_ISR( xADCTaskWoken );
    }
}
/*-----------------------------------------------------------*/

void
uDMAErrorHandler(void)
{
    /* The interrupt handler for uDMA errors.  This interrupt will occur if the
     * uDMA encounters a bus error while trying to perform a transfer.  This
     * handler just increments a counter if an error occurs. */

uint32_t ui32Status;

    /* Check for uDMA error bit */
    ui32Status = uDMAErrorStatusGet();

    /* If there is a uDMA error, then clear the error and increment
     * the error counter.
     */
    if(ui32Status)
    {
        uDMAErrorStatusClear();
        g_ui32uDMAErrCount++;
    }
}
/*-----------------------------------------------------------*/

void
uDMAIntHandler(void)
{
    /* The interrupt handler for uDMA interrupts from the memory channel.  This
     * interrupt will increment a counter, and then restart another memory
     * transfer. */

    g_ui32BadISR++;
}

